import Validator from "..";
import { Service } from 'typedi';
import { FeeValidatorAdapter } from "../../Database/adapters/feeAdapter";
import { NextFunction, Request, Response } from "express";
import { badRequest } from "../../Utils/api_response";
import { Pay_refValidatorAdapter } from "../../Database/adapters/pay_refAdapter";
import { AspirantValidatorAdapter } from "../../Database/adapters/aspirantAdapter";
import { Category, Fee_Type, NID_Stream, PROF_Stream, Stream } from "../../types/enum";

@Service()
class FeeValidation extends Validator {
  constructor(
    private readonly aspirantValAdapter: AspirantValidatorAdapter,
    private readonly feevalAdapter: FeeValidatorAdapter,
    private readonly pay_refValAdapter: Pay_refValidatorAdapter
  ) {
    super()
  }

  public validateFormPayment = this.validate({
    stream: {
      in: 'params',
      isIn: { options: [Object.values(PROF_Stream)] }
    },
    split: {
      in: 'query',
      isBoolean: true,
      toBoolean: true,
      custom: {
        options: async split => {
          const allowed = await this.feevalAdapter.DBIsFeeSplittable(Fee_Type.FORM, Category.PROFESSIONAL);

          if (split && !allowed) throw new Error('Sorry this fee is can\'t be splitted');
        }
      }
    },
    email: {
      in: ["body"],
      isEmail: true,
      toLowerCase: true,
      custom: {
        options: async (email) => {
          const aspirant = await this.aspirantValAdapter.DBCheckApplicationEmail(email);

          if (aspirant) throw new Error("Email already exists!")
        }
      }
    },
    first_name: {
      in: ["body"],
      isString: true,
      notEmpty: true,
    },
    last_name: {
      in: ["body"],
      isString: true,
      notEmpty: true,
    },
    middle_name: {
      in: ["body"],
      isString: true,
      optional: true
    },
  })

  public validateGetProfAcceptanceFee = this.validate({
    stream: {
      in: ['params'],
      isIn: { options: [Object.values(Stream)] },
      custom: {
        options: async (stream, { req }) => {
          // check if paid for appllication form
          const { auth_user } = req.body
          const aspirant = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(auth_user.email);

          if (!aspirant) throw new Error('You are not eligible too pay for this fee at the moment')
        }
      }
    }
  })

  // NID

  public validateGetAcceptanceFee = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { auth_user } = req.body;

      const aspirant = await this.aspirantValAdapter.DBGetAcceptedAspirant(auth_user.jamb_reg);

      if (!aspirant) throw new Error(`You can\'t perform this action. Kindlyy wait till you're given admission`);

      return next()
    } catch (error) {
      console.log(error)
      return badRequest(res, null, "Something went wrong");
    }
  }

  // BOTH

  public validateFeeVerification = this.validate({
    reference: {
      in: 'query',
      isString: true,
      custom: {
        options: async (reference, { req }) => {
          const used = await this.pay_refValAdapter.DBReferenceUsed(reference);
          console.log(used);
          if (used) throw new Error('Invalid payment, rf');
        }
      }
    },
    fee_id: {
      in: 'body',
      isInt: true,
      custom: {
        options: async (fee_id) => {
          const fee = await this.feevalAdapter.DBFeeExist(fee_id);

          if (!fee) throw new Error('Invalid fee');
        }
      }
    }
  })

  public validateAcceptancePayment = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { auth_user } = req.body;
      const { stream } = req.params as { stream: Stream };
      const { split } = req.query as unknown as { split: boolean };

      if (stream && !(Object.values(Stream).includes(stream))) throw new Error('Invalid stream');

      // let aspirant;
      // if (stream) {
      //   aspirant = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(auth_user.email);
      // } else {
      //   aspirant = await this.aspirantValAdapter.DBGetAcceptedAspirant(auth_user.jamb_reg);
      // }

      let aspirant = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(req.body.auth_user.email);

      if (!aspirant) aspirant = await this.aspirantValAdapter.DBGetAcceptedAspirant(req.body.auth_user.jamb_reg);

      if (!aspirant) throw new Error('You are not eligible to pay the acceptance fee')

      if (aspirant && aspirant.paid) return badRequest(res, null, "You already paid the acceptance fee, proceed to pay the school fee if you're yet to do so");

      // const category = auth_user.stream ? Category.PROFESSIONAL : Category.STANDARD
      const category = (auth_user.stream === PROF_Stream.BASIC || auth_user.stream === PROF_Stream.ADVANCED) ? Category.PROFESSIONAL : auth_user.stream === NID_Stream.FULL ? Category.FULL : Category.PART
      const allowed = await this.feevalAdapter.DBIsFeeSplittable(Fee_Type.ACCEPTANCE, category);

      // if (split as boolean && !allowed) throw new Error('Sorry this fee can\'t be splitted');
      const spt = split === true
      if (spt as boolean && !allowed) throw new Error('Sorry this fee can\'t be splitted');

      return next();
    } catch (error) {
      console.log(error)
      return badRequest(res, null, "Something went wrong");
    }
  }

  public validateGetSchoolFee = this.validate({
    stream: {
      in: ['params'],
      isIn: { options: [Object.values(Stream)] },
      custom: {
        options: async (stream) => {
          // check if paid for appllication form
        }
      }
    },
    split: {
      in: 'query',
      isBoolean: true,
      toBoolean: true,
      custom: {
        options: async (split, { req }) => {
          // const strm = req.params?.stream || Stream.STANDARD
          const allowed = await this.feevalAdapter.DBIsFeeSplittable(Fee_Type.TUITION, req.params?.stream);

          if (split && !allowed) throw new Error('Sorry this fee is can\'t be splitted');
        }
      }
    },
  })

  public validatePaySchoolFeeAndIDCard = this.validate({
    stream: {
      in: 'params',
      isIn: { options: [Object.values(Category)] },
      optional: true
    },
    split: {
      in: ['query'],
      isBoolean: true,
      toBoolean: true,
      custom: {
        options: async (split, { req }) => {
          // check if paid for acceptance fee
          // let aspi;
          // if (req.params?.stream) {
          //   aspi = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(req.body.auth_user.email)
          // } else {
          //   aspi = await this.aspirantValAdapter.DBGetAcceptedAspirant(req.body.auth_user.jamb_reg)
          // }

          let aspi = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(req.body.auth_user.email);

          if (!aspi) aspi = await this.aspirantValAdapter.DBGetAcceptedAspirant(req.body.auth_user.jamb_reg);

          if (!aspi.paid) throw new Error('You can only pay this fee after you clearing the acceptance fee or payng the both together');

          // const strm = req.params?.stream || Stream.STANDARD
          const allowed = await this.feevalAdapter.DBIsFeeSplittable(Fee_Type.TUITION, req.body.auth_user.stream);

          if (split && !allowed) throw new Error('Sorry this fee is can\'t be splitted');
        }
      }
    }
  })

  public validateSchoolFeeAndIDCardVerification = this.validate({
    reference: {
      in: 'query',
      isString: true,
      custom: {
        options: async (reference, { req }) => {
          const used = await this.pay_refValAdapter.DBReferenceUsed(reference);
          console.log(used);
          if (used) throw new Error('Invalid payment, rf');
        }
      }
    },
    fee_ids: {
      in: 'body',
      // isArray: true,
      // notEmpty: true,
      // isLength: { options: { min: 2, max: 2 } },
      isString: true,
      custom: {
        options: async (fee_ids) => {
          // '1,5'
          fee_ids.split(',').map((id: string) => +id).map(async (fee_id: number) => {
            const fee = await this.feevalAdapter.DBFeeExist(fee_id);

            if (!fee) throw new Error('Invalid fee');
          })
          // fee_ids.map(async (fee_id: number) => {
          //   const fee = await this.feevalAdapter.DBFeeExist(fee_id);

          //   if (!fee) throw new Error('Invalid fee');
          // })

        }
      }
    }
  })

  public validatePayAcceptanceSchoolFeeAndIDCard = this.validate({
    stream: {
      in: 'params',
      isIn: { options: [Object.values(Category)] },
      optional: true
    },
    split: {
      in: ['query'],
      isBoolean: true,
      toBoolean: true,
      custom: {
        options: async (split, { req }) => {
          // let aspirant;

          // if (req.params?.stream) {
          //   aspirant = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(req.body.auth_user.email);
          // } else {
          //   aspirant = await this.aspirantValAdapter.DBGetAcceptedAspirant(req.body.auth_user.jamb_reg);
          // }

          let aspirant = await this.aspirantValAdapter.DBGetAcceptedProfAspirant(req.body.auth_user.email);

          if (!aspirant) aspirant = await this.aspirantValAdapter.DBGetAcceptedAspirant(req.body.auth_user.jamb_reg);

          if (!aspirant) throw new Error('You are not eligible to pay the acceptance fee')

          if (aspirant && aspirant.paid) throw new Error("You already paid the acceptance fee, proceed to pay the school fee if you're yet to do so");

          // const strm = req.params?.stream || Stream.STANDARD
          const allowed = await this.feevalAdapter.DBIsFeeSplittable(Fee_Type.TUITION, req.body.auth_user.stream);

          if (split && !allowed) throw new Error('Sorry this fee is can\'t be splitted');
        }
      }
    }
  })

  public validateAcceptanceSchoolFeeAndIDCardVerification = this.validate({
    reference: {
      in: 'query',
      isString: true,
      custom: {
        options: async (reference, { req }) => {
          const used = await this.pay_refValAdapter.DBReferenceUsed(reference);
          console.log(used);
          if (used) throw new Error('Invalid payment, rf');
        }
      }
    },
    fee_ids: {
      in: 'body',
      isString: true,
      custom: {
        options: async (fee_ids) => {
          fee_ids.split(',').map((id: string) => +id).map(async (fee_id: number) => {
            const fee = await this.feevalAdapter.DBFeeExist(fee_id);

            if (!fee) throw new Error('Invalid fee');
          })
        }
      }
    }
  })

  public validateSchoolFeePayment = async (req: Request, res: Response, next: NextFunction) => {
    try {
      const { auth_user } = req.body;

      // const checkOutstanding = 
      return next();
    } catch (error) {
      return badRequest(res, null, "Something went wrong");
    }
  }

  // STUDENTTT

  public validateGetOutstandingFee = this.validate({
    // returning: {
    //   in: 'query',
    //   isBoolean: true,
    //   toBoolean: true,
    // },
  })

  public validateVerifyFeePayment = this.validate({
    reference: {
      in: 'query',
      isString: true,
      custom: {
        options: async (reference) => {
          const used = await this.pay_refValAdapter.DBReferenceUsed(reference);

          if (used) throw new Error('Invalid payment, rf');
        }
      }
    },
    fee_ids: {
      in: 'body',
      isString: true,
      custom: {
        options: async (fee_ids) => {
          fee_ids.split(',').map((id: string) => +id).map(async (fee_id: number) => {
            const fee = await this.feevalAdapter.DBFeeExist(fee_id);

            if (!fee) throw new Error('Invalid fee');
          })
        }
      }
    },
    residual: {
      in: ['query'],
      isBoolean: true,
      toBoolean: true
    }
  })
}

export default FeeValidation;