
import multer from 'multer';
import { NextFunction, Request, Response } from 'express';
import path from 'path';
import fs from 'fs';
import fsPromises from 'fs/promises';


// Define allowed file types
const allowedFileTypes = ['application/pdf', 'image/jpeg', 'image/png'];

// const storage = multer.memoryStorage();
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        // const uploadDir = path.join(__dirname, process.env.UPLOAD_PATH ?? 'uploads'); // Change to your desired directory
        const uploadDir = path.join(process.env.UPLOAD_PATH ?? 'public/uploads'); // Change to your desired directory
        // Ensure that the directory exists
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir);
        }
        cb(null, uploadDir);
    },
    // filename: (req, file, cb) => {
    //     // const uniqueName = `${Date.now()}-${file.originalname}`;
    //     const uniqueName = `${Date.now()}-${file.fieldname}.${file.mimetype.split('/')[1]}`;
    //     cb(null, uniqueName); // You can use any naming strategy here
    // }
    filename: (req: any, file, cb) => {
        const uniqueName = `${Date.now()}-${file.fieldname}.${file.mimetype.split('/')[1]}`;
        // const filePath = path.join(uploadPath, uniqueName);
        const filePath = path.join(process.env.UPLOAD_PATH ?? 'public/uploads', uniqueName);
        req.uploadedFiles?.push(filePath); // Track file path
        
        cb(null, uniqueName);
      }
});

// Define the fileFilter function to check file type
const fileFilter = (req: any, file: any, cb: any) => {
    if (allowedFileTypes.includes(file.mimetype)) {
        cb(null, true);
    } else {
        cb(new Error('Invalid file type. Only PDF, JPEG, and PNG are allowed.'), false);
    }
};

export const multerUpload = multer({
    storage: storage,
    limits: {
        fileSize: 500 * 1024 // Limit file size to 500KB
    },
    fileFilter: fileFilter
});

export const trackUploadedFiles = (req: any, res: Response, next: NextFunction) => {
    req.uploadedFiles = [];
    next();
};

// Custom error handling middleware for file size
export const fileUploadErrorHandler = (err: any, req: any, res: Response, next: NextFunction) => {
    // Log the error for debugging
    console.error(err, 'INSIDE multer file');

    // Delete all uploaded files in case of an error
    if (req.uploadedFiles && req.uploadedFiles.length > 0) {
        deleteFiles(req.uploadedFiles);
    }

    // Handle Multer's specific errors
    if (err instanceof multer.MulterError) {
        switch (err.code) {
            case 'LIMIT_FILE_SIZE':
                return res.status(400).json({ error: `File size exceeds the 500KB limit at ${err.field}`, message: `File size exceeds the 500KB limit at ${err.field}` });
            default:
                return res.status(400).json({ error: `Multer error: ${err.message}`, message: err.message });
        }
    }

    // Handle other types of errors (e.g., invalid file type)
    if (err.message) {
        return res.status(400).json({ error: err.message, message: err.message });
    }

    // If the error is unhandled, pass it to the next error handler
    next(err);
};

const directoryExists = async (dir: string) => {
    try {
        await fsPromises.access(dir);
        return true;
    } catch {
        return false;
    }
};

export const deleteFiles = async (filePaths: string[]) => {
    filePaths.forEach((filePath) => {
    try {
        if (fs.existsSync(filePath)) {
            fs.unlinkSync(filePath);
            console.log(`Deleted file: ${filePath}`);
        } else {
            console.warn(`File does not exist: ${filePath}`);
        }
    } catch (err) {
        console.error(`Error deleting file ${filePath}:`, err);
    }
    });
};