import { Service } from "typedi";
import knex, { BaseAdapter } from ".";
import { ETables } from "../../types/db.types";
import { Level, Semester, Stream } from "../../types/enum";
import { getDate } from "date-fns";

@Service()
class StudentAdapter extends BaseAdapter {
    constructor() {
        super()
    }
    public DBCreateStudent = async (auth_user: any, password: string, adm_no: string, paid: boolean, level: Level, diet_id?: number) => {
        try {
            return await knex.transaction(async trx => {
                // const { id, first_name, last_name, middle_name, email, DOB, stream, description, ...enrlData } = auth_user
                const { first_name, last_name, middle_name, email, DOB, gender, phone, stream, jamb_reg, qualification, workplace, years_of_exp, programme_id, session_id } = auth_user;

                const dateOnly = new Date(DOB).toISOString().split('T')[0]
                console.log({ first_name, last_name, middle_name, email, password, DOB:dateOnly, gender, phone, stream: stream, adm_no, paid, matric_no: adm_no.split('/').pop(), level, programme_id })

                const [student] = await trx(ETables.STUDENT).insert({ first_name, last_name, middle_name, email, password, DOB:dateOnly, gender, phone, stream: stream, adm_no, paid, matric_no: adm_no.split('/').pop(), level });

                await trx(ETables.ENROLLMENT).insert({ qualification, workplace, years_of_exp, stream: stream, enrollment_date: new Date(), completion_date: new Date(), jamb_reg, programme_id, student_id: student, session_id, diet_id });

                return student;
            })

        } catch (error) {
            return this.catchError(error);
        }
    }

    public getProgrammeByStream = async (stream: Stream) => {
        try {
            const programme = await knex.select('*').from(ETables.PROGRAMME).where('stream', stream).first();

            return programme;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public getDietById = async (id: number) => {
        try {
            const diet = await knex.select('*').from(ETables.DIET).where('id', id).first();

            return diet;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAcademicSessions = async () => {
        try {
            const sessions = await knex.select('*').from(ETables.ACADEMIC_SESSION);

            return sessions;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetStudent = async (matric_no: string) => {
        try {
            const student = await knex.select('*').from(ETables.STUDENT).where("matric_no", matric_no).first();

            return student;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetProfile = async (matric_no: string) => {
        try {
            const profile = await knex.select([
                "std.id as id",
                "std.first_name as first_name",
                "std.last_name as last_name",
                "std.middle_name as middle_name",
                "std.email as email",
                "std.adm_no as adm_no",
                "std.matric_no as matric_no",
                "std.phone as phone",
                "std.DOB as DOB",
                "std.gender as gender",
                "std.GP as GP",
                "std.CGP as CGP",
                "std.level as level",
                "std.stream as stream",
                "enrl.entry_session as entry_session",
                "enrl.jamb_reg as jamb_reg",
                "enrl.enrollment_date as enrollment_date",
                "enrl.completion_date as completion_date",
                "enrl.programme_id as programme_id",
                "prg.duration as duration",
                "prg.name as programme",
                "prg.description as programme_description",
                // "prg.stream as stream",
                "diet.number as diet",
                "diet.year as diet_year",
            ])
                .from({ std: ETables.STUDENT })
                .join({ enrl: ETables.ENROLLMENT }, "enrl.student_id", "std.id")
                .leftJoin({ prg: ETables.PROGRAMME }, "prg.id", "enrl.programme_id")
                .leftJoin({ diet: ETables.DIET }, 'diet.id', 'enrl.diet_id')
                .where("matric_no", matric_no).first();

            // console.log(profile);

            return profile
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetResult = async (student_id: number, level: Level, session_id: number, semester: Semester) => {
        try {
            const results = await knex.select([
                "reg.CA as CA",
                "reg.exam as exam",
                "reg.grade as grade",
                "reg.current_level as level",
                "reg.course_id as course_id",
                "course.code as code",
                "course.title as title",
                "course.description as description",
                "course.unit as unit",
                "course.type as type",
                "course.semester as semester",
                "reg.session_id as session_id"
            ])
            .from({reg: ETables.COURSE_REG})
            .join({ course: ETables.COURSES }, 'course.id', 'reg.course_id')
            .where('student_id', student_id).andWhere('current_level', level).andWhere('session_id', session_id )

            return results;
        } catch (error) {
            return this.catchError(error);
        }
    }
    
    public DBGetWeight = async (score: number) => {
        try {
            console.log(score)
            const weight = await knex.select('*').from(ETables.GRADE).where("mark_min", '<=', score).andWhere("mark_max", ">=", score).first();
            console.log(weight)
            return weight;
        } catch (error) {
            return this.catchError(error);
        }
    }
    
    public DGGetCummulativeResullt = async (student_id: number) => {
        try {
            const results = await knex.select([
                "reg.grade as grade",
                "reg.current_level as level",
                "reg.course_id as course_id",
                "course.code as code",
                "course.title as title",
                "course.description as description",
                "course.unit as unit",
                "course.type as type",
                "course.semester as semester",
                "reg.session_id as session_id"
            ])
            .from({reg: ETables.COURSE_REG})
            .join({ course: ETables.COURSES }, 'course.id', 'reg.course_id')
            .where('student_id', student_id)

            return results;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetCOD = async (cgpa: number) => {
        try {
            console.log(cgpa, 'cgpa')
            const cod = await knex.select('*').from(ETables.CGPA_GRADE).where("mark_min", '<=', cgpa).andWhere("mark_max", ">=", cgpa).first();
            console.log(cod)
            return cod || 'Fail';
        } catch (error) {
            return this.catchError(error);
        }
    }
















    public DBGetEnrollmentData = async (student_id: number) => {
        try {
            const data = await knex.select('*').from(ETables.ENROLLMENT).where('student_id', student_id).first();

            return data;
        } catch (error) {
            return this.catchError(error);
        }
    }
}

@Service()
export class StudentValidatorAdapter extends BaseAdapter {
    constructor() {
        super()
    }

    public DBGetStudent = async (matric_no: string) => {
        try {
            const student = await knex.select('*').from(ETables.STUDENT).where("matric_no", matric_no).first();

            return student;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBIsValidSession = async (id: number) => {
        try {
            const session = await knex.select('*').from(ETables.ACADEMIC_SESSION).where("id", id).first();

            return !!session;
        } catch (error) {
            return this.catchError(error);
        }
    }
}

export default StudentAdapter;