import { Service } from "typedi";
import knex, { BaseAdapter } from ".";
import { ETables } from "../../types/db.types";
import { Level, Semester, Course_Type, Stream } from "../../types/enum";

@Service()
class CourseAdapter extends BaseAdapter {
    constructor() {
        super()
    }

    public DBGetCurrentSessionAndDiet = async () => {
        try {
            const seesion = await knex.select('*').from(ETables.ACADEMIC_SESSION).where("is_current", true).first();

            const diet = await knex.select('*').from(ETables.DIET).where("is_current", true).andWhere('session_id', seesion.id).first();

            seesion['current_diet'] = diet;

            return seesion;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAllCourses= async (stream: Stream) => {
        try {
            const courses = await knex.select('*').from(ETables.COURSES).where('stream', stream);

            return courses;
        } catch (error) {
            return this.catchError(error);   
        }
    }

    public DBGetCourseById= async (id: number) => {
        try {
            const course = await knex.select('*').from(ETables.COURSES).where('id', id).first();

            return course;
        } catch (error) {
            return this.catchError(error);   
        }
    }

    public DBGetMyCourses= async (user_id: number) => {
        try {
            const courses = await knex.select([
                "reg.*",
                "course.id as course_id",
                "course.code as code",
                "course.title as title",
                "course.description as description",
                "course.unit as unit",
                "course.type as type",
                "course.semester as semester",
            ])
            .from({ reg: ETables.COURSE_REG})
            .join({ course: ETables.COURSES}, 'course.id', 'reg.course_id')
            .where('student_id', user_id);

            return courses;
        } catch (error) {
            return this.catchError(error);   
        }
    }

    public DBRegister = async (user_id: number, ids: number[], session_id: number, current_level: number) => {
        try {
            const reg = [];

            for(let id of ids) {
                const obj = {
                    current_level,
                    course_id: id,
                    student_id: user_id,
                    session_id
                };
                
                reg.push(obj);
            };

            await knex(ETables.COURSE_REG).insert(reg);
        } catch (error) {
            return this.catchError(error);   
        }
    }

    public DBDropCourse = async (student_id: number, course_id: number, session_id: number) => {
        try {
            await knex(ETables.COURSE_REG).where("student_id", student_id).andWhere({ course_id, session_id }).delete();
        } catch (error) {
            return this.catchError(error);
        }
    }
}

@Service()
export class CourseValidatorAdapter extends BaseAdapter {
    constructor() {
        super()
    }

    public DBGetCourseById = async (id: number) => {
        try {
            const course = await knex.select(['id', 'code']).from(ETables.COURSES).where("id", id).first();

            return course;
        } catch (error) {
            return this.catchError(error);   
        }
    }

    public DBHasCourseForCurrentSession = async (id: number, student_id: number, session_id: number) => {
        try {
            const course = await knex.select('*').from(ETables.COURSE_REG).where("session_id", session_id).andWhere("student_id", student_id).andWhere("course_id", id).first();

            return !!course;
        } catch (error) {
            return this.catchError(error);
        }
    }

    public DBGetAllCourses = async (stream: Stream) => {
        try {
            const course = await knex.select('*').from(ETables.COURSES).where('stream', stream);

            return course;
        } catch (error) {
            return this.catchError(error);   
        }
    }

    public DBCourseAlreadyAssignedForTheSession = async (id: number, session_id: number) => {
        try {
            const course = await knex.select('*').from(ETables.ASSIGNED_COURSES).where('course_id', id).andWhere('session_id', session_id).first();

            return course;
        } catch (error) {
            return this.catchError(error);   
        }
    }
}

export default CourseAdapter;