import { Request, Response } from "express";
import { Inject, Service } from "typedi";
import IndexController from "..";
import StudentAdapter from "../../Database/adapters/studentAdapter";
import { generateToken } from "../../Utils/jwt";
import { created, success, successAction } from "../../Utils/api_response";
import { Level, Semester } from "../../types/enum";
import { genAdmissionLetterPDF, generateExamSlip } from "../../Utils/pdf/jspdf";

@Service()
class StudentController extends IndexController {
    constructor(
        @Inject() private readonly studentAdapter: StudentAdapter
    ) {
        super()
    }

    public getAcademicSessions = async (req: Request, res: Response) => {
        try {
            const sessions = await this.studentAdapter.DBGetAcademicSessions();
            console.log(sessions)
            return success(res, sessions);
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public login = async (req: Request, res: Response) => {
        try {
            const { matric_no } = req.body;
            const student = await this.studentAdapter.DBGetStudent(matric_no);

            const token = generateToken({ matric_no, id: student.id });

            return success(res, { token }, "Logged in successfully");
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getProfile = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;

            const me = await this.studentAdapter.DBGetProfile(auth_user.matric_no);

            // me['current_session'] = session.name;
            me.current_session = session.name;

            return success(res, me);
        } catch (error) {
            return this.catchError(error, res);
        }
    };

    public getResult = async (req: Request, res: Response) => {
        try {
            const { auth_user, session } = req.body;
            const { level, session_id, semester } = req.query as { level: Level, session_id: string, semester: Semester };

            let total_obatined_grade = 0;
            let total_units_taken = 0

            if (!+session_id) {
                const overall_result = await this.studentAdapter.DGGetCummulativeResullt(auth_user.id);

                for (let result of overall_result) {
                    const unit = result.unit;
                    const score = result.grade

                    const wt = await this.studentAdapter.DBGetWeight(score);
                    console.log(wt.weight, 'weight')
                    const g = wt.weight * unit

                    total_obatined_grade += g
                    total_units_taken += unit
                }

                const GPA = (total_obatined_grade / total_units_taken).toFixed(2);

                const CGPA = (total_obatined_grade / total_units_taken).toFixed(2);

                const COD = await this.studentAdapter.DBGetCOD(+CGPA);

                return success(res, { results: overall_result, GPA, CGPA, COD });
            }

            const results = await this.studentAdapter.DBGetResult(auth_user.id, level, +session_id, semester);

            for (let result of results) {
                const unit = result.unit;
                const score = result.grade

                const wt = await this.studentAdapter.DBGetWeight(score);
                console.log(wt.weight, 'weight')
                const g = wt.weight * unit

                total_obatined_grade += g
                total_units_taken += unit
            }

            console.log(total_obatined_grade, 'total_obatined_grade');
            console.log(total_units_taken, 'total_units_taken');

            const GPA = (total_obatined_grade / total_units_taken).toFixed(2);

            const overall_result = await this.studentAdapter.DGGetCummulativeResullt(auth_user.id);

            for (let result of overall_result) {
                const unit = result.unit;
                const score = result.grade

                const wt = await this.studentAdapter.DBGetWeight(score);
                console.log(wt.weight, 'weight')
                const g = wt.weight * unit

                total_obatined_grade += g
                total_units_taken += unit
            }

            const CGPA = (total_obatined_grade / total_units_taken).toFixed(2);

            const COD = await this.studentAdapter.DBGetCOD(+CGPA);

            return success(res, { results, GPA, CGPA, COD });
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public generateAdmissionLetter = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body
            const output = genAdmissionLetterPDF(auth_user.first_name, auth_user.stream, new Date());

            // Set the content type to application/pdf
            res.setHeader('Content-Type', 'application/pdf');
            // Suggest download file name
            res.setHeader('Content-Disposition', `attachment; filename=admission-letter-${auth_user.first_name.replace(/\s+/g, '-')}.pdf`);

            // Send the PDF as a buffer
            res.send(Buffer.from(output));
        } catch (error) {
            return this.catchError(error, res);
        }
    }

    public getExamSlip = async (req: Request, res: Response) => {
        try {
            const { auth_user } = req.body;

            const output = generateExamSlip(auth_user);

            res.setHeader('Content-Type', 'application/pdf');
            res.setHeader('Content-Disposition', `attachment; filename=Exam_slip-${auth_user.first_name.replace(/\s+/g, '-')}.pdf`);
            res.send(Buffer.from(output));
        } catch (error) {
            return this.catchError(error, res);
        }
    }
};

export default StudentController;