"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const bcrypt = tslib_1.__importStar(require("bcryptjs"));
const __1 = tslib_1.__importDefault(require(".."));
const aspirantAdapter_1 = require("../../Database/adapters/aspirantAdapter");
const jwt_1 = require("../../Utils/jwt");
const enum_1 = require("../../types/enum");
const studentAdapter_1 = require("../../Database/adapters/studentAdapter");
const typedi_1 = require("typedi");
const feeAdapter_1 = require("../../Database/adapters/feeAdapter");
const api_response_1 = require("../../Utils/api_response");
let AspirantValidation = class AspirantValidation extends __1.default {
    constructor(aspirantValAdapter, studentValAdapter, feeValAdapter) {
        super();
        this.aspirantValAdapter = aspirantValAdapter;
        this.studentValAdapter = studentValAdapter;
        this.feeValAdapter = feeValAdapter;
        this.validateProfSignup = this.validate({
            email: {
                in: ["body"],
                isEmail: true,
                toLowerCase: true,
                custom: {
                    options: (email) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const aspirant = yield this.aspirantValAdapter.DBCheckApplicationEmail(email);
                        if (aspirant)
                            throw new Error("Email already in use!");
                    })
                }
            },
            // password: {
            //     in: ["body"],
            //     isString: true,
            //     isLength: { options: { min: 8 } },
            // },
            first_name: {
                in: ["body"],
                isString: true,
                notEmpty: true,
            },
            last_name: {
                in: ["body"],
                isString: true,
                notEmpty: true,
            },
            middle_name: {
                in: ["body"],
                isString: true,
                // optional: true
            },
            phone: {
                in: ["body"],
                isMobilePhone: true,
                // optional: true,
                errorMessage: 'Invalid phone number'
            },
            stream: {
                in: ['body'],
                isIn: { options: [Object.values(enum_1.PROF_Stream)] },
            }
        });
        this.emailVerificationValidator = this.validate({
            // email: {
            //     in: ["body"],
            //     isEmail: true,
            //     toLowerCase: true,
            // },
            token: {
                in: ["query"],
                isString: true,
                custom: {
                    options: (token, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        try {
                            const { email, first_name, last_name, middle_name, stream } = (0, jwt_1.decodeProfAspirantFormToken)(token);
                            const str = stream;
                            if (!(Object.values(enum_1.PROF_Stream).includes(stream)))
                                throw new Error('Invalid stream provided');
                            const openToAdmission = yield this.feeValAdapter.DBIsProgrammeOpenToAdmission(str);
                            if (!openToAdmission)
                                throw new Error('This programme is not open to appication at the moment, please check back later');
                            req.body['aspirant'] = {
                                email,
                                stream,
                                first_name,
                                last_name,
                                middle_name
                            };
                            // console.log(req.body.aspirant)
                        }
                        catch (error) {
                            throw new Error("Verification code expired!");
                        }
                    })
                }
            },
            // stream: {
            //     in: 'body',
            //     isString: true,
            //     isIn: { options: [Object.values(PROF_Stream)] }
            // }
        });
        this.validateLogin = this.validate({
            email: {
                in: ["body"],
                isEmail: true,
                toLowerCase: true,
            },
            password: {
                in: ["body"],
                isString: true,
                notEmpty: true,
                // isLength: { options: { min: 8 } },
                custom: {
                    options: (password, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const aspirant = yield this.aspirantValAdapter.DBGetProfAspirant(req.body.email);
                        if (!aspirant || !bcrypt.compareSync(password, aspirant.password))
                            throw new Error("Invalid credentials!");
                    })
                }
            }
        });
        this.validateCheckApplicationStatus = this.validate({
            programme_name: {
                in: ["params"],
                isString: true,
                isIn: { options: [Object.values(enum_1.PROGRAMMES)] }
            },
        });
        this.validateApplication = this.validate({
            middle_name: {
                in: ["body"],
                isString: true,
            },
            email: {
                in: ["body"],
                isEmail: true,
                toLowerCase: true,
                custom: {
                    options: (email, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const { auth_user } = req.body;
                        // console.log(req.body, 'req.body')
                        const aspirant = yield this.aspirantValAdapter.DBCheckApplicationEmail(email);
                        // check if paid
                        const form_fee = yield this.feeValAdapter.DBGetFeeByNameAndCategory('Form', enum_1.Category.PROFESSIONAL);
                        const has_paid_form_fee = yield this.feeValAdapter.DBHasPaidFee(form_fee.id, auth_user.id);
                        if (!has_paid_form_fee)
                            throw new Error('You not eligible to this resource, kindly pay the application form to continue');
                        if (aspirant.status)
                            throw new Error('You can only fill the application form once.');
                        if (aspirant.email !== auth_user.email)
                            throw new Error("Invalid email");
                        // if (aspirant.stream !== auth_user.stream) throw new Error('Invalid stream');
                    })
                }
            },
            DOB: {
                in: ['body'],
                isString: true,
                isISO8601: { errorMessage: "invalid iso" },
            },
            gender: {
                in: 'body',
                isIn: { options: [Object.values(enum_1.Gender)] }
            },
            stream: {
                in: ['params'],
                isIn: { options: [Object.values(enum_1.PROF_Stream)] },
            },
            qualification: {
                in: 'body',
                isString: true
            },
            workplace: {
                in: 'body',
                isString: true
            },
            years_of_exp: {
                in: 'body',
                isInt: true
            },
        });
        // NID
        this.validatePUTMEScreening = this.validate({
            first_name: {
                in: ["body"],
                isString: true,
                notEmpty: true,
            },
            last_name: {
                in: ["body"],
                isString: true,
                notEmpty: true,
            },
            middle_name: {
                in: ["body"],
                isString: true,
                optional: true
            },
            email: {
                in: ["body"],
                isEmail: true,
                toLowerCase: true,
                custom: {
                    options: (email) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const aspirant = yield this.aspirantValAdapter.DBCheckScreeningEmail(email);
                        if (aspirant)
                            throw new Error("Email already exits!");
                    })
                }
            },
            // stream: {
            //     in: ['body'],
            //     isIn: { options: [Object.values(PROF_Stream)] },
            // },
            jamb_reg: {
                in: 'body',
                isString: true,
                isLength: { options: { min: 14 } },
                custom: {
                    options: (jamb_reg, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const exist = yield this.aspirantValAdapter.DBCheckJambReg(jamb_reg);
                        if (exist)
                            throw new Error('Account already exist');
                        console.log(req.file, 'file');
                        console.log(req.files, 'files');
                        const { o_level, jamb_result } = req.files;
                        if (!o_level || !jamb_result)
                            throw new Error('Please upload the required documents');
                    })
                }
            },
            DOB: {
                in: ['body'],
                isString: true,
                isISO8601: { errorMessage: "invalid iso" },
            },
            gender: {
                in: 'body',
                isIn: { options: [Object.values(enum_1.Gender)] }
            },
        });
        this.validateNIDLogin = this.validate({
            jamb_reg: {
                in: 'body',
                isString: true,
                isLength: { options: { min: 14 } },
            },
            password: {
                in: ['body'],
                isString: true,
                custom: {
                    options: (password, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const aspirant = yield this.aspirantValAdapter.DBGetAspirant(req.body.jamb_reg);
                        if (!aspirant || !bcrypt.compareSync(password, aspirant.password))
                            throw new Error("Invalid credentials!");
                    })
                }
            }
        });
        this.validateUploadAdmissionLetter = (req, res, next) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                // const { buffer, mimetype } = req.file!
                // if (!buffer) throw new Error('Kindly upload your admission letter')
                if (!req.file)
                    throw new Error('Kindly upload your admission letter');
                return next();
            }
            catch (error) {
                console.log(error);
                return (0, api_response_1.badRequest)(res, null, "Something went wrong");
            }
        });
        // public validateUpload = () = async (req: Request, res: Response, next: NextFunction) => {
        this.validateUpload = (req, res, next) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const files = req.files;
                const { o_level, passport } = files;
                // Check if files are uploaded
                if (!o_level || !passport || !o_level[0] || !passport[0])
                    throw new Error('Kindly upload the necessary files');
                // const allowedFileTypes = ['application/pdf', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'application/msword'];
                // // Check the MIME types of the uploaded files
                // const o_levelMimetype = o_level[0].mimetype;
                // const passportMimetype = passport[0].mimetype;
                // if (!allowedFileTypes.includes(o_levelMimetype) || !allowedFileTypes.includes(passportMimetype)) {
                //     throw new Error("Invalid file type");
                // }
                return next();
            }
            catch (error) {
                console.log(error);
                return (0, api_response_1.badRequest)(res, null, "Something went wrong");
            }
        });
        // NID PART TIME
        this.validatePartTimeApplication = this.validate({
            first_name: {
                in: ["body"],
                isString: true,
                notEmpty: true,
            },
            last_name: {
                in: ["body"],
                isString: true,
                notEmpty: true,
            },
            middle_name: {
                in: ["body"],
                isString: true,
                optional: true
            },
            email: {
                in: ["body"],
                isEmail: true,
                toLowerCase: true,
                custom: {
                    options: (email, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const aspirant = yield this.aspirantValAdapter.DBCheckScreeningEmail(email);
                        if (aspirant)
                            throw new Error("Email already exits!");
                    })
                }
            },
            // stream: {
            //     in: ['body'],
            //     isIn: { options: [Object.values(PROF_Stream)] },
            // },
            DOB: {
                in: ['body'],
                isString: true,
                isISO8601: { errorMessage: "invalid iso" },
            },
            gender: {
                in: 'body',
                isIn: { options: [Object.values(enum_1.Gender)] }
            },
            state_of_origin: {
                in: 'body',
                isString: true
            },
            lga: {
                in: 'body',
                isString: true
            },
            phone: {
                in: 'body',
                isMobilePhone: { options: 'any' }
            },
            address: {
                in: 'body',
                isString: true
            },
            // manual_o_level
        });
        this.validatePartLogin = this.validate({
            email: {
                in: 'body',
                isString: true,
                isEmail: true,
                toLowerCase: true
            },
            password: {
                in: ['body'],
                isString: true,
                custom: {
                    options: (password, { req }) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        const aspirant = yield this.aspirantValAdapter.DBGetAspirantByEmail(req.body.email);
                        if (!aspirant || !bcrypt.compareSync(password, aspirant.password))
                            throw new Error("Invalid credentials!");
                    })
                }
            }
        });
    }
};
AspirantValidation = tslib_1.__decorate([
    (0, typedi_1.Service)(),
    tslib_1.__metadata("design:paramtypes", [aspirantAdapter_1.AspirantValidatorAdapter,
        studentAdapter_1.StudentValidatorAdapter,
        feeAdapter_1.FeeValidatorAdapter])
], AspirantValidation);
exports.default = AspirantValidation;
