"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteFiles = exports.fileUploadErrorHandler = exports.trackUploadedFiles = exports.multerUpload = void 0;
const tslib_1 = require("tslib");
const multer_1 = tslib_1.__importDefault(require("multer"));
const path_1 = tslib_1.__importDefault(require("path"));
const fs_1 = tslib_1.__importDefault(require("fs"));
const promises_1 = tslib_1.__importDefault(require("fs/promises"));
// Define allowed file types
const allowedFileTypes = ['application/pdf', 'image/jpeg', 'image/png'];
// const storage = multer.memoryStorage();
const storage = multer_1.default.diskStorage({
    destination: (req, file, cb) => {
        var _a;
        // const uploadDir = path.join(__dirname, process.env.UPLOAD_PATH ?? 'uploads'); // Change to your desired directory
        const uploadDir = path_1.default.join((_a = process.env.UPLOAD_PATH) !== null && _a !== void 0 ? _a : 'public/uploads'); // Change to your desired directory
        // Ensure that the directory exists
        if (!fs_1.default.existsSync(uploadDir)) {
            fs_1.default.mkdirSync(uploadDir);
        }
        cb(null, uploadDir);
    },
    // filename: (req, file, cb) => {
    //     // const uniqueName = `${Date.now()}-${file.originalname}`;
    //     const uniqueName = `${Date.now()}-${file.fieldname}.${file.mimetype.split('/')[1]}`;
    //     cb(null, uniqueName); // You can use any naming strategy here
    // }
    filename: (req, file, cb) => {
        var _a, _b;
        const uniqueName = `${Date.now()}-${file.fieldname}.${file.mimetype.split('/')[1]}`;
        // const filePath = path.join(uploadPath, uniqueName);
        const filePath = path_1.default.join((_a = process.env.UPLOAD_PATH) !== null && _a !== void 0 ? _a : 'public/uploads', uniqueName);
        (_b = req.uploadedFiles) === null || _b === void 0 ? void 0 : _b.push(filePath); // Track file path
        cb(null, uniqueName);
    }
});
// Define the fileFilter function to check file type
const fileFilter = (req, file, cb) => {
    if (allowedFileTypes.includes(file.mimetype)) {
        cb(null, true);
    }
    else {
        cb(new Error('Invalid file type. Only PDF, JPEG, and PNG are allowed.'), false);
    }
};
exports.multerUpload = (0, multer_1.default)({
    storage: storage,
    limits: {
        fileSize: 500 * 1024 // Limit file size to 500KB
    },
    fileFilter: fileFilter
});
const trackUploadedFiles = (req, res, next) => {
    req.uploadedFiles = [];
    next();
};
exports.trackUploadedFiles = trackUploadedFiles;
// Custom error handling middleware for file size
const fileUploadErrorHandler = (err, req, res, next) => {
    // Log the error for debugging
    console.error(err, 'INSIDE multer file');
    // Delete all uploaded files in case of an error
    if (req.uploadedFiles && req.uploadedFiles.length > 0) {
        (0, exports.deleteFiles)(req.uploadedFiles);
    }
    // Handle Multer's specific errors
    if (err instanceof multer_1.default.MulterError) {
        switch (err.code) {
            case 'LIMIT_FILE_SIZE':
                return res.status(400).json({ error: `File size exceeds the 500KB limit at ${err.field}`, message: `File size exceeds the 500KB limit at ${err.field}` });
            default:
                return res.status(400).json({ error: `Multer error: ${err.message}`, message: err.message });
        }
    }
    // Handle other types of errors (e.g., invalid file type)
    if (err.message) {
        return res.status(400).json({ error: err.message, message: err.message });
    }
    // If the error is unhandled, pass it to the next error handler
    next(err);
};
exports.fileUploadErrorHandler = fileUploadErrorHandler;
const directoryExists = (dir) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    try {
        yield promises_1.default.access(dir);
        return true;
    }
    catch (_a) {
        return false;
    }
});
const deleteFiles = (filePaths) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    filePaths.forEach((filePath) => {
        try {
            if (fs_1.default.existsSync(filePath)) {
                fs_1.default.unlinkSync(filePath);
                console.log(`Deleted file: ${filePath}`);
            }
            else {
                console.warn(`File does not exist: ${filePath}`);
            }
        }
        catch (err) {
            console.error(`Error deleting file ${filePath}:`, err);
        }
    });
});
exports.deleteFiles = deleteFiles;
