"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resizeImage = exports.getCompletionDateV2 = exports.getCompletionDate = exports.generateApplicationNo = exports.calculateCharges = exports.drawTable = exports.excelSerialDateToJSDate = exports.generateTransactionID = exports.generateRegNo = exports.encodeLastName = exports.fileStore = exports.pinGenerator = void 0;
const tslib_1 = require("tslib");
const date_fns_1 = require("date-fns");
const adapters_1 = tslib_1.__importDefault(require("../Database/adapters"));
const db_types_1 = require("../types/db.types");
const bcryptjs_1 = tslib_1.__importDefault(require("bcryptjs"));
const enum_1 = require("../types/enum");
const sharp_1 = tslib_1.__importDefault(require("sharp"));
const pinGenerator = (l) => {
    const p = l - 1;
    const value = Math.floor(Math.pow(10, p) + Math.random() * (9 * Math.pow(10, p)));
    return value.toString();
};
exports.pinGenerator = pinGenerator;
exports.fileStore = new Map();
const encodeLastName = (last_name) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const salt = yield bcryptjs_1.default.genSalt(10);
    return bcryptjs_1.default.hashSync(last_name.toLowerCase(), salt);
});
exports.encodeLastName = encodeLastName;
const generateRegNo = (programme, diet) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    // const programme = await knex.select('*').from(ETables.PROGRAMME).where("id", programme_id).first();
    const students = yield adapters_1.default.select(['email', 'adm_no']).from(db_types_1.ETables.STUDENT);
    let prog;
    switch (programme.stream) {
        case enum_1.Stream.PART:
            prog = 'PAR';
            break;
        case enum_1.Stream.FULL:
            prog = 'FUL';
            break;
        case 'basic':
            prog = 'BAS';
            break;
        case 'advanced':
            prog = 'ADV';
            break;
        default:
            break;
    }
    if (!students.length)
        return `TNM/${prog}/${diet.number}${diet.year % 100}/0001`;
    const nos = [];
    const stds = students.map(s => s.adm_no);
    stds.map(no => {
        const splitted = no.split('/').pop();
        console.log(splitted, 'splitted');
        // let num = splitted[splitted.length - 1];
        nos.push(+splitted);
    });
    let last = Math.max(...nos) + 1;
    let formattedLast = String(last).padStart(4, '0');
    return `TNM/${prog}/${diet.number}${diet.year % 100}/${formattedLast}`;
});
exports.generateRegNo = generateRegNo;
const generateTransactionID = () => {
    const rand = Math.floor(Math.pow(10, 2) + Math.random() * 10);
    const t = Date.now();
    return rand + t;
};
exports.generateTransactionID = generateTransactionID;
function excelSerialDateToJSDate(serialDate) {
    // Excel serial date starts on January 1, 1900. JavaScript's Date object starts on January 1, 1970.
    // Additionally, Excel incorrectly treats February 29, 1900, as a valid date, which it is not
    // because 1900 is not a leap year. Hence, any serial date after 59 is off by one. 
    // To compensate, we subtract 1 for dates after February 28, 1900.
    const excelEpoch = new Date(1900, 0, -1);
    const jsDate = new Date(excelEpoch.getTime() + serialDate * 86400000);
    return jsDate;
}
exports.excelSerialDateToJSDate = excelSerialDateToJSDate;
const drawTable = (doc, data) => {
    const tableTop = 100;
    const columnWidths = [200, 300]; // Adjust based on your needs
    const rowHeight = 20;
    let currentRowHeight = tableTop;
    // Draw headers
    doc.fontSize(12).fillColor('black').text('Name', 50, tableTop);
    doc.text('Courses', 50 + columnWidths[0], tableTop);
    // Draw each row
    data.forEach((row, i) => {
        currentRowHeight += rowHeight;
        doc.fontSize(10).fillColor('black').text(row.name, 50, currentRowHeight);
        doc.text(row.courses.join(', '), 50 + columnWidths[0], currentRowHeight);
    });
};
exports.drawTable = drawTable;
const calculateCharges = (amount) => {
    let fee;
    if (amount <= 2500)
        return fee = 1.5 / 100 * amount;
    else {
        fee = (1.5 / 100 * amount) + 100;
        if (fee > 2000)
            return 2000;
        return fee;
    }
};
exports.calculateCharges = calculateCharges;
const generateApplicationNo = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    // const programme = await knex.select('*').from(ETables.PROGRAMME).where("id", programme_id).first();
    const aspirants = yield adapters_1.default.select(['email', 'reg_no']).from(db_types_1.ETables.ASPIRANT);
    const prof_aspirants = yield adapters_1.default.select(['email', 'reg_no']).from(db_types_1.ETables.PROF_ASPIRANT);
    const combined = aspirants.concat(prof_aspirants);
    const currYear = new Date().getFullYear();
    if (!combined.length)
        return `TNM${currYear}${currYear + 1}/0001`;
    const nos = [];
    const applicants = combined.map(s => s.reg_no);
    applicants.map(no => {
        const splitted = no.split('/').pop();
        console.log(splitted, 'splitted');
        // let num = splitted[splitted.length - 1];
        nos.push(+splitted);
    });
    let last = Math.max(...nos) + 1;
    let formattedLast = String(last).padStart(4, '0');
    return `TNM${currYear}${currYear + 1}/${formattedLast}`;
});
exports.generateApplicationNo = generateApplicationNo;
const getCompletionDate = (duration) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const [number, timeframe] = duration.split(' ');
    const num = parseInt(number, 10);
    if (isNaN(num)) {
        throw new Error('Invalid duration number');
    }
    const currentDate = new Date();
    let completion_date;
    if (timeframe === 'weeks') {
        completion_date = new Date(currentDate);
        completion_date.setDate(currentDate.getDate() + num * 7);
    }
    else if (timeframe === 'years') {
        completion_date = new Date(currentDate);
        completion_date.setFullYear(currentDate.getFullYear() + num);
    }
    else {
        throw new Error('Invalid timeframe. Only "weeks" or "years" are supported.');
    }
    // Format the completion date in the "YYYY-MM-DD" format
    const formattedDate = completion_date.toISOString().split('T')[0];
    return formattedDate;
});
exports.getCompletionDate = getCompletionDate;
const getCompletionDateV2 = (duration) => {
    const [number, timeframe] = duration.split(' ');
    const num = parseInt(number, 10);
    if (isNaN(num)) {
        throw new Error('Invalid duration number');
    }
    const currentDate = new Date();
    let completionDate;
    if (timeframe === 'weeks') {
        completionDate = (0, date_fns_1.addWeeks)(currentDate, num);
    }
    else if (timeframe === 'years') {
        completionDate = (0, date_fns_1.addYears)(currentDate, num);
    }
    else {
        throw new Error('Invalid timeframe. Only "weeks" or "years" are supported.');
    }
    const formattedDate = (0, date_fns_1.format)(completionDate, 'yyyy-MM-dd');
    return formattedDate;
};
exports.getCompletionDateV2 = getCompletionDateV2;
const resizeImage = (imagePath, width, height, shouldResize = true) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    if (!shouldResize) {
        const originalImageBuffer = yield (0, sharp_1.default)(imagePath).jpeg({ quality: 70 }).toBuffer();
        return originalImageBuffer.toString('base64');
    }
    const resizedImageBuffer = yield (0, sharp_1.default)(imagePath)
        .resize(width, height)
        .jpeg({ quality: 70 })
        .toBuffer();
    return resizedImageBuffer.toString('base64');
});
exports.resizeImage = resizeImage;
